<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor Overview Calendar Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */
class Elementor_WPSBC_Overview_Calendar_Widget extends \Elementor\Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Overview Calendar widget name.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'wpsbc_overview_calendar';
    }

    /**
     * Get widget title.
     *
     * Retrieve Overview Calendar widget title.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget title.
     */
    public function get_title()
    {
        return esc_html__('WP Simple Booking Calendar - Overview Calendar', 'wp-simple-booking-calendar');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Overview Calendar widget icon.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-table-of-contents';
    }

    /**
     * Get custom help URL.
     *
     * Retrieve a URL where the user can get more information about the widget.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget help URL.
     */
    public function get_custom_help_url()
    {
        return 'https://www.wpsimplebookingcalendar.com/';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Overview Calendar widget belongs to.
     *
     * @since 1.0.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['wp-simple-booking-calendar'];
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the Overview Calendar widget belongs to.
     *
     * @since 1.0.0
     * @access public
     * @return array Widget keywords.
     */
    public function get_keywords()
    {
        return ['Overview', 'Calendar', 'Multiple', 'WP Simple Booking Calendar', 'wpsimplebookingcalendar'];
    }

    /**
     * Register Overview Calendar widget controls.
     *
     * Add input fields to allow the user to customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'calendar',
            [
                'label' => esc_html__('Calendars', 'wp-simple-booking-calendar'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $calendars = wpsbc_get_calendars(array('status' => 'active'));
        $calendarDropdown = array();
        foreach ($calendars as $calendar) {
            $calendarDropdown[$calendar->get('id')] = $calendar->get('name');
        }

        $this->add_control(
            'calendars_type',
            [
                'label' => esc_html__('Display', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => array(
                    'all' => __('All Calendars', 'wp-simple-booking-calendar'),
                    'selected' => __('Selected Calendars', 'wp-simple-booking-calendar'),
                ),
                'default' => 'all',
            ]
        );

        $this->add_control(
            'calendars',
            [
                'label' => esc_html__('Calendars', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT2,
                'options' => $calendarDropdown,
                'multiple' => true,
                'condition' => array('calendars_type' => 'selected'),
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'calendar_options',
            [
                'label' => esc_html__('Calendar Options', 'wp-simple-booking-calendar'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'legend',
            [
                'label' => esc_html__('Display Legend', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => array(
                    'yes' => __('Yes', 'wp-simple-booking-calendar'),
                    'no' => __('No', 'wp-simple-booking-calendar'),
                ),
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'legend_position',
            [
                'label' => esc_html__('Legend Position', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => array(
                    'top' => __('Top', 'wp-simple-booking-calendar'),
                    'bottom' => __('Bottom', 'wp-simple-booking-calendar'),
                ),
                'default' => 'top',
            ]
        );

        $this->add_control(
            'start_year',
            [
                'label' => esc_html__('Start Year', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [0 => __('Current Year', 'wp-simple-booking-calendar')] + array_combine(range(date('Y'), date('Y') + 10), range(date('Y'), date('Y') + 10)),
                'default' => '0',
            ]
        );

        $this->add_control(
            'start_month',
            [
                'label' => esc_html__('Start Month', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => array(
                    0 => __('Current Month', 'wp-simple-booking-calendar'),
                    1 => __('January', 'wp-simple-booking-calendar'),
                    2 => __('February', 'wp-simple-booking-calendar'),
                    3 => __('March', 'wp-simple-booking-calendar'),
                    4 => __('April', 'wp-simple-booking-calendar'),
                    5 => __('May', 'wp-simple-booking-calendar'),
                    6 => __('June', 'wp-simple-booking-calendar'),
                    7 => __('July', 'wp-simple-booking-calendar'),
                    8 => __('August', 'wp-simple-booking-calendar'),
                    9 => __('September', 'wp-simple-booking-calendar'),
                    10 => __('October', 'wp-simple-booking-calendar'),
                    11 => __('November', 'wp-simple-booking-calendar'),
                    12 => __('December', 'wp-simple-booking-calendar'),
                ),
                'default' => '0',
            ]
        );

        $this->add_control(
            'history',
            [
                'label' => esc_html__('Show History', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => array(
                    '1' => __('Display booking history', 'wp-simple-booking-calendar'),
                    '2' => __('Replace booking history with the default legend item', 'wp-simple-booking-calendar'),
                    '3' => __('Use the Booking History Color from the Settings', 'wp-simple-booking-calendar'),
                ),
                'default' => '1',
            ]
        );

        $this->add_control(
            'tooltip',
            [
                'label' => esc_html__('Display Tooltips', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => array(
                    '1' => __('No', 'wp-simple-booking-calendar'),
                    '2' => __('Yes', 'wp-simple-booking-calendar'),
                    '3' => __('Yes, with red indicator', 'wp-simple-booking-calendar'),
                ),
                'default' => '1',
            ]
        );

        $this->add_control(
            'weeknumbers',
            [
                'label' => esc_html__('Show Weekday Abbreviations', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => array(
                    'yes' => __('Yes', 'wp-simple-booking-calendar'),
                    'no' => __('No', 'wp-simple-booking-calendar'),
                ),
                'default' => 'no',
            ]
        );

        $settings = get_option('wpsbc_settings');
        $languages = wpsbc_get_languages();
        $languagesDropdown = array('auto' => 'Auto');

        if (!empty($settings['active_languages'])) {

            foreach ($settings['active_languages'] as $key => $code) {

                if (empty($languages[$code])) {
                    continue;
                }

                $languagesDropdown[$code] = $languages[$code];

            }

        }

        $this->add_control(
            'language',
            [
                'label' => esc_html__('Language', 'wp-simple-booking-calendar'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => $languagesDropdown,
                'default' => 'auto',
            ]
        );

        $this->end_controls_section();

    }

    /**
     * Render Overview Calendar widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {

        $settings = $this->get_settings_for_display();

        if ($settings['calendars_type'] == 'all') {
            $settings['calendars'] = 'all';
        } else {
            $settings['calendars'] = implode(',', $settings['calendars']);

            if (empty($settings['calendars'])) {
                $settings['calendars'] = 'all';
            }

        }

        // Execute the shortcode
        echo WPSBC_Shortcodes::calendar_overview($settings);

    }

}
