<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;


/**
 * Add the General Settings fields needed by this module
 *
 * @param array $settings
 *
 */
function wpsbc_add_user_restrictions_general_settings_fields( $settings ) {

	if( ! current_user_can( 'manage_options' ) )
		return;

	include 'views/view-settings-tab-general-fields.php';

}
add_action( 'wpsbc_submenu_page_settings_tab_general_bottom', 'wpsbc_add_user_restrictions_general_settings_fields' );


/**
 * Add the Edit Calendar fields needed by this module
 *
 * @param WPSBC_Calendar
 *
 */
function wpsbc_add_user_restrictions_edit_calendar_fields( $calendar ) {

	if( ! wpsbc_current_user_can_edit_plugin() )
		return;

	include 'views/view-edit-calendar-main-fields.php';

}
add_action( 'wpsbc_view_edit_calendar_main', 'wpsbc_add_user_restrictions_edit_calendar_fields' );


/**
 * Save the Edit Calendar fields
 *
 * @param array $post_data
 *
 */
function wpsbc_save_user_restrictions_edit_calendar_fields( $post_data ) {

	if( empty( $post_data['form_data'] ) )
		return;

	$form_data = $post_data['form_data'];

	if( empty( $form_data['calendar_id'] ) )
		return;

	if( empty( $form_data['wpsbc_token_calendar_user_premissions'] ) || ! wp_verify_nonce( $form_data['wpsbc_token_calendar_user_premissions'], 'wpsbc_token_calendar_user_premissions' ) )
		return;

	$calendar_id = (int)$form_data['calendar_id'];

	// Delete the user permission meta
	wpsbc_delete_calendar_meta( $calendar_id, 'user_permission' );

	// Update calendar user permissions meta if the value exists
	if( ! empty( $form_data['calendar_user_permissions'] ) && is_array( $form_data['calendar_user_permissions'] ) ) {

		foreach( $form_data['calendar_user_permissions'] as $user_id )
			wpsbc_add_calendar_meta( $calendar_id, 'user_permission', $user_id );

	}

}
add_action( 'wpsbc_save_calendar_data', 'wpsbc_save_user_restrictions_edit_calendar_fields' );


/**
 * Determines whether the current user has capabilities to edit plugin settings
 *
 * This is determined by checking if the current user has one of the roles from the Settings page
 *
 * @return bool
 *
 */
function wpsbc_current_user_can_edit_plugin() {

	if( current_user_can( 'manage_options' ) )
		return true;

	$settings = get_option( 'wpsbc_settings', array() );
	$user 	  = wp_get_current_user();

	$user_role_permissions = ( ! empty( $settings['user_role_permissions'] ) ? $settings['user_role_permissions'] : array() );

	foreach( $user_role_permissions as $user_role ) {

		if( in_array( $user_role, $user->roles ) )
			return true;

	}

	return false;

}


/**
 * Determines whether the current user has capabilities to edit any calendars
 *
 * @return bool
 *
 */
function wpsbc_current_user_can_edit_any_calendars() {

	if( current_user_can( 'manage_options' ) )
		return true;

	$calendars = wpsbc_get_calendars();
	$user 	   = wp_get_current_user();

	foreach( $calendars as $calendar ) {

		if( wpsbc_current_user_can_edit_calendar( $calendar->get('id') ) )
			return true;

	}

	return false;

}


/**
 * Determines whether the current user has capabilities to edit the given calendar
 *
 * @param int $calendar_id
 *
 * @return bool
 *
 */
function wpsbc_current_user_can_edit_calendar( $calendar_id ) {

	$user = wp_get_current_user();
	$user_permissions = wpsbc_get_calendar_meta( $calendar_id, 'user_permission' );

	if( empty( $user_permissions ) )
		$user_permissions = array();

	if( in_array( $user->ID, $user_permissions ) )
		return true;

	return false;

}


/**
 * Modifies the permisions for the main plugin admin page and submenu pages
 *
 * @param string $capability
 *
 * @return string
 *
 */
function wpsbc_set_submenu_page_capabilities( $capability = 'manage_options' ) {

	if( current_user_can( 'manage_options' ) )
		return 'manage_options';

	if( wpsbc_current_user_can_edit_plugin() )
		return 'read';

	// If on add new calendar page
	if( ! empty( $_GET['subpage'] ) && $_GET['subpage'] == 'add-calendar' ) {

		if( wpsbc_current_user_can_edit_any_calendars() )
			return 'manage_options';

	}

	// If on edit calendar page
	if( ! empty( $_GET['calendar_id'] ) ) {

		if( wpsbc_current_user_can_edit_calendar( (int)$_GET['calendar_id'] ) )
			return 'read';

	} else {

		if( wpsbc_current_user_can_edit_any_calendars() )
			return 'read';

	}

	return $capability;

}
add_filter( 'wpsbc_menu_page_capability', 'wpsbc_set_submenu_page_capabilities' );
add_filter( 'wpsbc_submenu_page_capability_calendars', 'wpsbc_set_submenu_page_capabilities' );


/**
 * Remove calendars from the wpsbc_get_calendars returned value
 *
 * @param array $calendars
 * @param array $args
 * @param bool  $count
 *
 * @return mixed array|int
 *
 */
function wpsbc_get_calendars_user_capabilities( $calendars, $args, $count = false ) {

	if( wpsbc_current_user_can_edit_plugin() )
	    return $calendars;

	if( ! is_admin() )
	    return $calendars;

	if( empty( $_GET['page'] ) || $_GET['page'] != 'wpsbc-calendars' )
	    return $calendars;

	// Get all calendar args
	$all_calendars_args = array(
	    'number' => -1,
	    'offset' => 0,
	    'status' => ( ! empty( $args['status'] ) ? $args['status'] : 'active' )
	);

	$all_calendars  = wp_simple_booking_calendar()->db['calendars']->get_calendars( $all_calendars_args );
	$user_calendars = array();

	foreach( $all_calendars as $calendar ) {

	    if( wpsbc_current_user_can_edit_calendar( $calendar->get('id') ) )
	        $user_calendars[] = $calendar;

	}

	// Handle the case where calendars are being present
	if( is_array( $calendars ) ) {

	    $calendars = array_slice( $user_calendars, ( ! empty( $args['offset'] ) ? $args['offset'] : 0 ), ( ! empty( $args['number'] ) ? $args['number'] : 20 ) );

	// Handle the case where the count of the calendars is present
	} else {

	    $calendars = count( $user_calendars );

	    if( $calendars < 0 )
	        $calendars = 0;

	}

	return $calendars;

}
add_filter( 'wpsbc_get_calendars', 'wpsbc_get_calendars_user_capabilities', 10, 3 );


/**
 * Remove the Calendar Table views for users that do not have complete access to the plugin
 *
 * @param array $views
 *
 * @return array
 *
 */
function wpsbc_list_table_calendars_remove_views( $views ) {

	if( wpsbc_current_user_can_edit_plugin() )
		return $views;

	return array();

}
add_filter( 'wpsbc_list_table_calendars_views', 'wpsbc_list_table_calendars_remove_views', 100 );


/**
 * Remove the Add New Calendar page title action button for users that do not have 
 * complete access to the plugin
 *
 */
function wpsbc_calendars_page_remove_title_actions() {

	if( ! wpsbc_current_user_can_edit_plugin() )
		echo '<style>.wpsbc-wrap-calendars .page-title-action { display: none; }</style>';

}
add_action( 'admin_head', 'wpsbc_calendars_page_remove_title_actions' );


/**
 * Removes the Calendar Table row actions for users that do not have complete access to the plugin
 *
 * @param array $actions
 * @param array $item
 *
 * @return array
 *
 */
function wpsbc_list_table_calendars_remove_row_actions( $actions, $item ) {

	if( wpsbc_current_user_can_edit_plugin() )
		return $actions;

	// Remove all actions if the calendar is in Trash
	if( $item['status'] == 'trash' )
		return array();

	// Remove the Trash option if it exists
	if( ! empty( $actions['trash'] ) )
		unset( $actions['trash'] );

	return $actions;

}
add_filter( 'wpsbc_list_table_calendars_row_actions', 'wpsbc_list_table_calendars_remove_row_actions', 100, 2 );